const { ipcRenderer } = require('electron');

let selectedDrive = null;
let selectedMethod = null;
let wipeResult = null;

const methods = [
    { id: 'quick', name: 'Quick Wipe', passes: 1, desc: 'Single pass (fastest)', security: 'Low' },
    { id: 'nist', name: 'NIST 800-88', passes: 1, desc: 'US Gov standard', security: 'Medium' },
    { id: 'random', name: 'Random Data', passes: 3, desc: '3-pass random', security: 'High' },
    { id: 'dod-5220', name: 'DoD 5220.22-M', passes: 3, desc: 'Military standard', security: 'High' },
    { id: 'vsitr', name: 'VSITR', passes: 7, desc: 'German standard', security: 'High' },
    { id: 'dod-ece', name: 'DoD 5220.22-M ECE', passes: 7, desc: 'DoD Extended', security: 'Maximum' },
    { id: 'gutmann', name: 'Gutmann Method', passes: 35, desc: 'Maximum security', security: 'Maximum' },
    { id: 'ata-secure', name: 'ATA Secure Erase', passes: 1, desc: 'Hardware SSD erase', security: 'High' }
];

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    renderMethods();
    refreshDrives();
});

function renderMethods() {
    const grid = document.getElementById('methodGrid');
    grid.innerHTML = methods.map(method => `
        <div class="method-card" onclick="selectMethod('${method.id}')" data-method="${method.id}">
            <div class="method-name">${method.name}</div>
            <span class="method-passes">${method.passes} pass${method.passes > 1 ? 'es' : ''}</span>
            <div class="method-desc">${method.desc}</div>
        </div>
    `).join('');
}

async function refreshDrives() {
    addLog('Scanning for connected drives...');

    const result = await ipcRenderer.invoke('get-drives');

    if (result.success) {
        renderDrives(result.drives);
        addLog(`Found ${result.drives.length} drive(s)`, 'success');
    } else {
        addLog(`Error: ${result.error}`, 'error');
    }
}

function renderDrives(drives) {
    const list = document.getElementById('driveList');

    if (drives.length === 0) {
        list.innerHTML = '<div class="drive-item" style="justify-content: center; color: #64748b;">No drives detected</div>';
        return;
    }

    list.innerHTML = drives.map(drive => {
        const size = formatBytes(drive.size || 0);
        const isSystem = drive.isSystem || drive.isRemovable === false;
        return `
            <div class="drive-item ${selectedDrive === drive.device ? 'selected' : ''}" 
                 onclick="selectDrive('${drive.device}', ${isSystem})"
                 data-device="${drive.device}">
                <div class="drive-icon">${drive.isUSB ? '🔌' : '💾'}</div>
                <div class="drive-info">
                    <div class="drive-name">${drive.description || 'Unknown Drive'}</div>
                    <div class="drive-details">
                        ${drive.device} • ${drive.busType || 'Unknown'} • 
                        ${isSystem ? '<span style="color: #ef4444;">⚠️ System Drive</span>' : 'External'}
                    </div>
                </div>
                <div class="drive-size">${size}</div>
            </div>
        `;
    }).join('');
}

function selectDrive(device, isSystem) {
    if (isSystem) {
        alert('⚠️ Warning: This appears to be a system drive. Wiping it will destroy your operating system!');
    }

    selectedDrive = device;
    document.querySelectorAll('.drive-item').forEach(el => el.classList.remove('selected'));
    document.querySelector(`[data-device="${device}"]`)?.classList.add('selected');
    updateWipeButton();
}

function selectMethod(methodId) {
    selectedMethod = methodId;
    document.querySelectorAll('.method-card').forEach(el => el.classList.remove('selected'));
    document.querySelector(`[data-method="${methodId}"]`)?.classList.add('selected');
    updateWipeButton();
}

function updateWipeButton() {
    const btn = document.getElementById('wipeBtn');
    btn.disabled = !(selectedDrive && selectedMethod);
}

function addLog(message, type = 'info') {
    const container = document.getElementById('logContainer');
    const entry = document.createElement('div');
    entry.className = `log-entry ${type}`;
    entry.textContent = `[${new Date().toLocaleTimeString()}] ${message}`;
    container.appendChild(entry);
    container.scrollTop = container.scrollHeight;
}

async function startWipe() {
    if (!selectedDrive || !selectedMethod) return;

    const method = methods.find(m => m.id === selectedMethod);
    const confirmed = confirm(
        `⚠️ DESTRUCTIVE ACTION ⚠️\n\n` +
        `Drive: ${selectedDrive}\n` +
        `Method: ${method.name} (${method.passes} passes)\n\n` +
        `This will PERMANENTLY destroy all data on this drive.\n` +
        `Are you absolutely sure?`
    );

    if (!confirmed) return;

    const btn = document.getElementById('wipeBtn');
    const progressContainer = document.getElementById('progressContainer');
    const progressFill = document.getElementById('progressFill');
    const progressText = document.getElementById('progressText');

    btn.disabled = true;
    progressContainer.classList.add('active');
    addLog(`Starting ${method.name} on ${selectedDrive}...`);

    // Listen for progress updates
    ipcRenderer.on('wipe-progress', (event, data) => {
        const percent = data.percent;
        progressFill.style.width = `${percent}%`;
        progressText.textContent = `Pass ${data.pass} of ${data.totalPasses} (${percent}%)`;
        addLog(`Pass ${data.pass}/${data.totalPasses} complete`);
    });

    const result = await ipcRenderer.invoke('wipe-drive', {
        device: selectedDrive,
        method: selectedMethod,
        passes: method.passes
    });

    if (result.success) {
        wipeResult = result.details;
        addLog(`✅ Wipe completed successfully in ${result.details.duration}s`, 'success');
        document.getElementById('certBtn').disabled = false;
        progressText.textContent = 'Complete! ✅';
    } else {
        addLog(`❌ Error: ${result.error}`, 'error');
        progressText.textContent = 'Failed ❌';
        btn.disabled = false;
    }
}

async function generateCertificate() {
    if (!wipeResult) return;

    const result = await ipcRenderer.invoke('generate-certificate', {
        device: selectedDrive,
        method: selectedMethod,
        ...wipeResult
    });

    if (result.success) {
        const cert = result.certificate;
        document.getElementById('certificate').classList.add('active');
        document.getElementById('certContent').innerHTML = `
            <p><strong>Certificate ID:</strong> ${cert.id}</p>
            <p><strong>Company:</strong> ${cert.company}</p>
            <p><strong>Date:</strong> ${cert.date}</p>
            <p><strong>Device:</strong> ${cert.device}</p>
            <p><strong>Method:</strong> ${cert.standard}</p>
            <p><strong>Technician:</strong> ${cert.technician}</p>
            <p><strong>Status:</strong> ✅ Verified</p>
        `;
        addLog('Certificate generated', 'success');
    }
}

async function saveCertificate() {
    const result = await ipcRenderer.invoke('show-save-dialog', {
        defaultPath: `PPSP-Certificate-${Date.now()}.pdf`,
        filters: [{ name: 'PDF', extensions: ['pdf'] }, { name: 'JSON', extensions: ['json'] }]
    });

    if (result && !result.canceled) {
        addLog(`Certificate saved to: ${result.filePath}`, 'success');
    }
}

function formatBytes(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}