// BARCODE & QR CODE IMPLEMENTATION FOR PPSP APPS
// Add this to your main.js or renderer process

const bwipjs = require('bwip-js');
const PDFDocument = require('pdfkit');
const fs = require('fs');
const path = require('path');

class CertificateGenerator {
  constructor(companyInfo) {
    this.company = companyInfo || {
      name: 'PPSP Recycling LLC',
      address: '9095 Elk Grove Blvd Suite B, Elk Grove, CA 95624',
      phone: '916-381-8304',
      email: 'info@ppsprecyclingllc.com',
      website: 'https://ppsprecyclingllc.com'
    };
  }

  async generateCertificate(wipeData) {
    const certId = `PPSP-${Date.now()}`;
    const timestamp = new Date().toLocaleString();

    // Generate barcodes
    const barcodeBuffer = await this.generateBarcode(certId);
    const qrBuffer = await this.generateQRCode(certId);

    // Create PDF
    const doc = new PDFDocument({ margin: 50 });
    const outputPath = path.join(__dirname, `../certificates/${certId}.pdf`);

    // Ensure directory exists
    fs.mkdirSync(path.dirname(outputPath), { recursive: true });

    doc.pipe(fs.createWriteStream(outputPath));

    // Header with barcode
    doc.image(barcodeBuffer, 50, 30, { width: 200, height: 50 });
    doc.fontSize(8).text(certId, 50, 85);

    // Title
    doc.fontSize(24).text('CERTIFICATE OF DATA DESTRUCTION', 0, 120, { align: 'center' });
    doc.moveDown();

    // Company info
    doc.fontSize(10).text(this.company.name, { align: 'center' });
    doc.text(this.company.address, { align: 'center' });
    doc.text(`Phone: ${this.company.phone}`, { align: 'center' });
    doc.moveDown(2);

    // Certificate details table
    const tableTop = 220;
    doc.fontSize(12);

    this.drawTable(doc, tableTop, [
      ['Certificate ID:', certId],
      ['Date & Time:', timestamp],
      ['Technician:', wipeData.technician || 'Authorized Technician'],
      ['', ''],
      ['Device Type:', wipeData.deviceType || 'Unknown'],
      ['Device Model:', wipeData.deviceModel || 'Unknown'],
      ['Serial Number:', wipeData.serialNumber || 'N/A'],
      ['Capacity:', this.formatBytes(wipeData.capacity)],
      ['', ''],
      ['Destruction Method:', wipeData.method],
      ['Security Standard:', wipeData.standard],
      ['Passes Completed:', wipeData.passes.toString()],
      ['Duration:', wipeData.duration],
      ['Verification:', 'PASSED ✓']
    ]);

    // QR Code for verification
    doc.image(qrBuffer, 400, 400, { width: 100, height: 100 });
    doc.fontSize(8).text('Scan to verify', 400, 505);

    // Signature lines
    doc.moveDown(4);
    doc.text('_'.repeat(40), 100, 550);
    doc.text('Authorized Signature', 100, 565);

    doc.text('_'.repeat(40), 350, 550);
    doc.text('Date', 350, 565);

    // Footer
    doc.fontSize(10).text(
      'This certificate confirms that the above device has been securely wiped in accordance with the stated standard.',
      50, 620, { align: 'center', width: 500 }
    );

    doc.fontSize(8).text(
      `© ${new Date().getFullYear()} ${this.company.name} - All rights reserved.`,
      50, 700, { align: 'center' }
    );

    doc.end();

    return {
      certificateId: certId,
      filePath: outputPath,
      barcodeData: barcodeBuffer.toString('base64'),
      qrData: qrBuffer.toString('base64')
    };
  }

  async generateBarcode(text) {
    return await bwipjs.toBuffer({
      bcid: 'code128',        // Barcode type
      text: text,             // Text to encode
      scale: 3,               // 3x scaling factor
      height: 10,             // Bar height, in millimeters
      includetext: true,      // Show human-readable text
      textxalign: 'center',   // Center text
      textfont: 'Arial',      // Font for text
      textsize: 10            // Font size
    });
  }

  async generateQRCode(certId) {
    const verifyUrl = `${this.company.website}/verify/${certId}`;

    return await bwipjs.toBuffer({
      bcid: 'qrcode',         // QR Code
      text: verifyUrl,        // URL to verification page
      scale: 5,               // Size
      eclevel: 'M'            // Error correction level (M = 15%)
    });
  }

  drawTable(doc, startY, rows) {
    let y = startY;
    const col1X = 100;
    const col2X = 250;

    rows.forEach(([label, value]) => {
      if (label === '' && value === '') {
        y += 10; // Empty row spacing
      } else {
        doc.font('Helvetica-Bold').text(label, col1X, y);
        doc.font('Helvetica').text(value, col2X, y);
        y += 20;
      }
    });
  }

  formatBytes(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  }

  async printCertificate(pdfPath, printerName) {
    // For direct printing (requires node-printer)
    const printer = require('node-printer');

    return new Promise((resolve, reject) => {
      printer.printFile({
        filename: pdfPath,
        printer: printerName || process.env.DEFAULT_PRINTER,
        success: resolve,
        error: reject
      });
    });
  }
}

// Integration with main app
ipcMain.handle('generate-certificate-with-barcode', async (event, wipeData) => {
  const generator = new CertificateGenerator();

  try {
    const result = await generator.generateCertificate(wipeData);

    // Also save metadata to database
    await saveToDatabase({
      certificateId: result.certificateId,
      filePath: result.filePath,
      wipeData: wipeData,
      createdAt: new Date().toISOString()
    });

    return {
      success: true,
      certificateId: result.certificateId,
      filePath: result.filePath,
      preview: result.barcodeData
    };
  } catch (error) {
    return { success: false, error: error.message };
  }
});

// For thermal label printers (Zebra, Dymo, etc.)
function generateZebraLabel(certId, deviceInfo) {
  // ZPL (Zebra Programming Language)
  const zpl = `
^XA
^PW400
^LL300
^FO20,20
^BCN,80,Y,N,N
^FD${certId}^FS
^FO20,110
^A0N,30,30
^FDPPSP Recycling LLC^FS
^FO20,150
^A0N,20,20
^FD${deviceInfo.model || 'Unknown Device'}^FS
^FO20,180
^A0N,20,20
^FDS/N: ${deviceInfo.serial || 'N/A'}^FS
^FO20,220
^A0N,25,25
^FD${new Date().toLocaleDateString()}^FS
^XZ
  `;

  return zpl;
}

module.exports = { CertificateGenerator, generateZebraLabel };
